"use client"

import { FadeIn } from "@/components/animations/fade-in";
import LoadingCircle from "@/components/icons/loading.circle";
import { DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import { Skeleton } from "@/components/ui/skeleton";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { DEFAULT_SWR_OPTIONS } from "@/config/swr";
import { potentialDateToDateStringFormatter } from "@/lib/formatters/date-formatter";
import { errorToStringFormatter } from "@/lib/formatters/error-formatters";
import { focusRing } from "@/lib/styles";
import { sleep } from "@/lib/timing";
import { cn, formatBytes } from "@/lib/utils";
import { TransformedAgency } from "@/types/transformers";
import { ComponentWithClassName, ComponentWithClassNameAndChildren } from "@/types/utils";
import { getProperties, getUrl } from "aws-amplify/storage";
import { Check, Download, Folders, X } from "lucide-react";
import Link from "next/link";
import { useState } from "react";
import useSWR from "swr";
import { StatusMetaDataSearchParamsOption } from "@/app/api/statuses/[agency]/metadata/route";
import { validatedFetch } from "@/lib/zod/validated-fetch";
import { statusMetadataGetResponseSchema } from "@/zod/domain.schema";

export function WorkflowDialogDownload({ data, className }: ComponentWithClassName<{ data: NonNullable<TransformedAgency> }>) {
  if (data.access !== "ALL") {
    return;
  }

  return (
    <>
      <DialogHeader className={className}>
        <DialogTitle>Download feed</DialogTitle>
        <DialogDescription>Currently viewing feed files for &apos;<span className="font-mono font-bold">{data.name}</span>&apos;.</DialogDescription>
      </DialogHeader>

      <ScrollArea className="h-fit w-full pr-3">
        <div className="grid pt-2 md:max-h-96 flex-col gap-4">
          <INTERNAL__DownloadFeed delay={0} option="preview" agency={data.gtfsId} />
          <INTERNAL__DownloadFeed delay={0.15} option="live" agency={data.gtfsId} />
          <Separator className="bg-transparent" />
        </div>
      </ScrollArea>
    </>
  )
}

function INTERNAL__DownloadFeed({
  delay = 0,
  className,
  children,
  option,
  agency,
}: ComponentWithClassNameAndChildren<{
  delay?: number,
  option: StatusMetaDataSearchParamsOption,
  agency: string,
}>) {
  const [downloading, setDownloading] = useState<'idle' | 'downloading' | 'finished'>('idle');
  const { isLoading, data } = useSWR(
    ['INTERNAL__FILE_DOWNLOAD_CONTENT', agency, option],
    async () => {
      try {
        const data = await validatedFetch({
          url: `/api/statuses/${agency}/metadata?option=${option}`,
          schema: statusMetadataGetResponseSchema,
        })

        if (!data || (typeof data === 'object' && !("metadata" in data))) {
          const error = 'ERROR: Unable to retrieve agencies s3 server file path';
          return error;
        }

        const path = data.metadata.paths.files
        const [properties, url] = await Promise.all([
          getProperties({ path: path, }),
          getUrl({ path: path }),
          sleep(1500),
        ])

        if (!properties || !url) {
          const error = `ERROR: Unable to retrieve properties or download url for ${agency}`;
          return error;
        }

        return {
          properties,
          url,
        };

      } catch (error) {
        if (error instanceof Error || error === 'string') {
          return errorToStringFormatter(error)
        }
      }
    },
    DEFAULT_SWR_OPTIONS
  )

  const handleDownloadingStates = async () => {
    setDownloading('downloading');
    try {
      await sleep(2500)
    } catch (error) {
      console.error('error', error)
    } finally {
      setDownloading('finished');
      setTimeout(() => setDownloading('idle'), 1500);
    }
  };

  const title = option === "preview" ? "Preview" : "Live"
  const isError = typeof data === 'string'
  const isDownloading = downloading !== 'idle'
  return (
    <FadeIn delay={delay}>
      <div className={cn("rounded-xl border border-border bg-background shadow-sm relative", className)}>
        <div className="grid grid-cols-[auto,_1fr] md:grid-cols-[auto,_1fr,_auto]">
          <div className="pl-4 grid place-items-center">
            <div
              className="flex size-9 shrink-0 items-center justify-center rounded-full border border-border"
              aria-hidden="true"
            >
              {isLoading ? (
                <LoadingCircle className="opacity-60 size-4 stroke-2" />
              ) : isError ? (
                <X className="opacity-60 size-4 stroke-2 text-destructive" />
              ) : (
                <Folders className="opacity-60 size-4 stroke-2 text-primary" />
              )}
            </div>
          </div>

          <div className="p-4 flex-1 grid grow gap-12 text-pretty">
            <div className="space-y-1">
              <p className="text-sm font-medium">{title} Feed</p>
              {isLoading ? (
                <>
                  <Skeleton className="w-full h-4" />
                  <Skeleton className="w-full h-4" />
                  <Skeleton className="w-full h-4" />
                </>
              ) : isError ? (
                <p className="text-xs text-muted-foreground line-clamp-5">Files for this environment are curently not available. One or more errors occured while retrieving file data. This may mean files have not been uploaded for this environment. If this does not seem right, contact your system administrator.</p>
              ) : (
                <div className="flex flex-col items-start gap-0.5 text-xs">
                  <div className="flex flex-row gap-1.5 items-center">
                    <p className="text-xs text-muted-foreground line-clamp-3">Size</p>
                    <p className="text-xs text-foreground line-clamp-3">{data?.properties.size && formatBytes(data.properties.size)}</p>
                  </div>

                  <div className="flex flex-row gap-1.5 items-center">
                    <p className="text-xs text-muted-foreground line-clamp-3">File</p>
                    <p className="text-xs text-foreground line-clamp-3">{data?.properties.path}</p>
                  </div>

                  <div className="flex flex-row gap-1.5 items-center">
                    <p className="text-xs text-muted-foreground line-clamp-3">Last modified</p>
                    <p className="text-xs text-foreground line-clamp-3">{potentialDateToDateStringFormatter(data?.properties.lastModified)}</p>
                  </div>
                </div>
              )}

              {children}
            </div>
          </div>

          {
            typeof data === 'object' && (
              <TooltipProvider delayDuration={0}>
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Link
                      href={data.url.url.href}
                      target="_blank"
                      onClick={handleDownloadingStates}
                      className={cn(
                        focusRing,
                        isDownloading || isLoading || isError ? "pointer-events-none opacity-50" : "",
                        "inline-flex items-center justify-center whitespace-nowrap rounded-md text-sm font-medium disabled:pointer-events-none disabled:opacity-50 border border-transparent !outline-none focus-visible:!z-[100] focus-visible:!border-s focus-visible:!border-e ring-offset-background transition-shadow focus-visible:!outline-none focus-visible:!ring-2 focus-visible:!ring-offset focus-visible:!border-primary focus-visible:!ring-primary/20",
                        "relative col-span-2 md:col-span-1 h-16 md:h-full w-full md:w-24 rounded-none max-md:rounded-b-xl max-md:border-t max-md:border-t-border md:rounded-e-xl md:border-s md:border-s-border bg-transparent",
                        "group focus-visible:!text-primary focus-visible:!border-primary ring-primary/20"
                      )}
                      aria-label={downloading === 'finished' ? "Downloaded" : "Download files to system"}
                    >
                      <div
                        className={cn(
                          "absolute transition-all",
                          downloading === 'finished' ? "scale-100 opacity-100" : "scale-0 opacity-0",
                        )}
                      >
                        <Check
                          className="stroke-emerald-500 size-3"
                          aria-hidden="true"
                        />
                      </div>

                      <div
                        className={cn(
                          "absolute transition-all",
                          downloading === 'downloading' ? "scale-100 opacity-100" : "scale-0 opacity-0",
                        )}
                      >
                        <LoadingCircle
                          className="stroke-muted-500 size-3"
                          aria-hidden="true"
                        />
                      </div>

                      <div className={"sr-only"}>
                        Download {title} feed files.
                      </div>

                      <div
                        className={cn(
                          "absolute transition-all",
                          downloading === 'idle' ? "scale-100 opacity-100" : "scale-0 opacity-0",
                        )}
                      >
                        <Download className="group-focus-visible:text-primary text-muted-foreground size-3" aria-hidden="true" />
                      </div>
                    </Link>
                  </TooltipTrigger>
                  <TooltipContent className="border border-input bg-popover px-2 py-1 text-xs text-muted-foreground">
                    Download {title} feed files.
                  </TooltipContent>
                </Tooltip>
              </TooltipProvider>
            )
          }
        </div>
      </div>
    </FadeIn>
  )
}
